<?php
/**
 * Module Library: WooCommerce Cart Products Module REST Controller class
 *
 * @package Divi
 * @since ??
 */

namespace ET\Builder\Packages\ModuleLibrary\WooCommerce\CartProducts;

if ( ! defined( 'ABSPATH' ) ) {
	die( 'Direct access forbidden.' );
}

use ET\Builder\Framework\Controllers\RESTController;
use ET\Builder\Framework\UserRole\UserRole;
use ET\Builder\Packages\WooCommerce\WooCommerceUtils;
use WP_Error;
use WP_REST_Request;
use WP_REST_Response;

/**
 * WooCommerce Cart Products REST Controller class.
 *
 * @since ??
 */
class WooCommerceCartProductsController extends RESTController {

	/**
	 * Retrieve the rendered HTML for the WooCommerce Cart Products module.
	 *
	 * @since ??
	 *
	 * @param WP_REST_Request $request REST request object.
	 *
	 * @return WP_REST_Response|WP_Error Returns the REST response object containing the rendered HTML.
	 *                                   If the request is invalid, a `WP_Error` object is returned.
	 */
	public static function index( WP_REST_Request $request ) {
		$common_required_params = WooCommerceUtils::validate_woocommerce_request_params( $request );

		// If the conditional tags are not set, the returned value is an error.
		if ( ! isset( $common_required_params['conditional_tags'] ) ) {
			return self::response_error( ...$common_required_params );
		}

		$show_update_cart_button = $request->get_param( 'showUpdateCartButton' ) ?? 'on';

		$args = [
			'show_update_cart_button' => $show_update_cart_button,
		];

		// Get the cart products HTML from the module.
		$html = WooCommerceCartProductsModule::get_cart_products( $args, $common_required_params['conditional_tags'] );

		$response = [
			'html' => $html,
		];

		return self::response_success( $response );
	}

	/**
	 * Get the arguments for the index action.
	 *
	 * This function returns an array that defines the arguments for the index action,
	 * which is used in the `register_rest_route()` function.
	 *
	 * @since ??
	 *
	 * @return array An array of arguments for the index action.
	 */
	public static function index_args(): array {
		return [
			'showUpdateCartButton' => [
				'type'              => 'string',
				'required'          => false,
				'default'           => 'on',
				'enum'              => [ 'on', 'off' ],
				'description'       => 'Whether to show the update cart button.',
				'sanitize_callback' => 'sanitize_text_field',
				'validate_callback' => function( $param ) {
					return in_array( $param, [ 'on', 'off' ], true );
				},
			],
			'conditional_tags'     => [
				'type'              => 'array',
				'required'          => false,
				'default'           => [],
				'description'       => 'Conditional tags for preview modes.',
				'items'             => [
					'type' => 'string',
				],
				'sanitize_callback' => function( $param ) {
					return array_map( 'sanitize_text_field', $param ?? [] );
				},
				'validate_callback' => function( $param ) {
					if ( [] === $param ) {
						return true;
					}

					if ( ! is_array( $param ) ) {
						return false;
					}

					// Ensure no empty strings in the array.
					foreach ( $param as $tag ) {
						if ( empty( $tag ) || ! is_string( $tag ) ) {
							return false;
						}
					}

					return true;
				},
			],
		];
	}

	/**
	 * Provides the permission status for the index action.
	 *
	 * @since ??
	 *
	 * @return bool Returns `true` if the current user has the permission to use the rest endpoint, otherwise `false`.
	 */
	public static function index_permission(): bool {
		return UserRole::can_current_user_use_visual_builder();
	}

}
