<?php
/**
 * Migration Manager Class
 *
 * This class manages the execution of multiple migrations in sequence
 * through a fluent interface.
 *
 * @since ??
 *
 * @package Divi
 */

namespace ET\Builder\Migration;

use ET\Builder\Migration\FlexboxMigration;
use ET\Builder\Migration\GlobalColorMigration;
use ET\Builder\Migration\AttributeMigration;
use ET\Builder\Migration\AttributePresetMigration;
use ET\Builder\Framework\Utility\StringUtility;

/**
 * Migration class for handling sequential migration execution.
 *
 * @since ??
 */
class Migration {

	/**
	 * Stores the migration classes to be executed.
	 *
	 * @since ??
	 *
	 * @var array
	 */
	private $_migrations = [];

	/**
	 * Apply a specific migration and return $this for method chaining.
	 *
	 * @since ??
	 *
	 * @param string $migration_class The migration class name to run.
	 * @return self
	 */
	public function apply( string $migration_class ): self {
		$this->_migrations[] = $migration_class;
		return $this;
	}

	/**
	 * Sort migrations by release version.
	 *
	 * @since ??
	 *
	 * @param array $migrations Array of migration class names.
	 *
	 * @return array Sorted array of migration class names.
	 */
	public function sort_migrations_by_version( array $migrations ): array {
		$sorted_migrations = $migrations;
		usort(
			$sorted_migrations,
			fn( $a, $b ) => StringUtility::version_compare( $a::get_release_version(), $b::get_release_version() )
		);

		return $sorted_migrations;
	}

	/**
	 * Execute all registered migrations in sequence, sorted by version.
	 *
	 * @since ??
	 *
	 * @return void
	 */
	public function execute(): void {
		// Sort migrations by release version before executing.
		$sorted_migrations = $this->sort_migrations_by_version( $this->_migrations );

		foreach ( $sorted_migrations as $migration_class ) {
			$migration_class::load();
		}
	}

	/**
	 * Execute only preset migrations from the registered migrations.
	 *
	 * This method finds all registered migrations that have a migrate_presets() method
	 * and executes them in version order. Used for contexts like D4 to D5 conversion
	 * where preset migrations need to run immediately.
	 *
	 * @since ??
	 *
	 * @return bool True if preset migrations were executed, false if none found.
	 */
	public function execute_preset_migrations(): bool {
		// Filter to get only migrations that have migrate_presets() method.
		$preset_migrations = array_filter(
			$this->_migrations,
			function( $migration_class ) {
				return method_exists( $migration_class, 'migrate_presets' );
			}
		);

		if ( empty( $preset_migrations ) ) {
			return false;
		}

		// Sort preset migrations by release version before executing.
		$sorted_migrations = $this->sort_migrations_by_version( $preset_migrations );

		// Execute each preset migration.
		foreach ( $sorted_migrations as $migration_class ) {
			$migration_class::migrate_presets();
		}

		return true;
	}

	/**
	 * Migrate individual preset data through all applicable migrations.
	 *
	 * This method applies all registered migrations that have a migrate_preset_item() method
	 * to a single preset item. Used for normalizing preset data during duplicate detection
	 * so that newly imported presets can be compared against existing migrated presets.
	 *
	 * @since ??
	 *
	 * @param array  $preset_item The preset item to migrate.
	 * @param string $module_name The module name for this preset.
	 *
	 * @return array The migrated preset item.
	 */
	public function migrate_preset_item( array $preset_item, string $module_name ): array {
		// Filter to get only migrations that have migrate_preset_item() method.
		$preset_item_migrations = array_filter(
			$this->_migrations,
			function( $migration_class ) {
				return method_exists( $migration_class, 'migrate_preset_item' );
			}
		);

		if ( empty( $preset_item_migrations ) ) {
			return $preset_item;
		}

		// Sort migrations by release version before executing.
		$sorted_migrations = $this->sort_migrations_by_version( $preset_item_migrations );

		// Apply each migration to the preset item.
		$migrated_preset = $preset_item;
		foreach ( $sorted_migrations as $migration_class ) {
			$migrated_preset = $migration_class::migrate_preset_item( $migrated_preset, $module_name );
		}

		return $migrated_preset;
	}

	/**
	 * Get a global instance of the migration manager.
	 *
	 * @since ??
	 *
	 * @return Migration The global migration instance.
	 */
	public static function get_instance(): Migration {
		static $instance = null;
		if ( null === $instance ) {
			$instance = self::_create_global_instance();
		}
		return $instance;
	}

	/**
	 * Create the global migration instance with all registered migrations.
	 *
	 * @since ??
	 *
	 * @return Migration The migration instance with all migrations registered.
	 */
	private static function _create_global_instance(): Migration {
		$migration = new Migration();

		// Register migrations here - single source of truth.
		$migration->apply( FlexboxMigration::class );
		$migration->apply( GlobalColorMigration::class );
		$migration->apply( AttributeMigration::class );
		$migration->apply( AttributePresetMigration::class );

		return $migration;
	}
}

// Execute all migrations using the global instance.
Migration::get_instance()->execute();
